/* Some routines shared between WEB programs.
   C) 2010, 2017, 2021--2022 Thierry Laronde <tlaronde@polynum.com>
   kerTeX Public Licence v1.0.

   This routine was extracted and adjusted from web2c V5.0C.
 */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "kertex.h"
#include "pascal.h"
#include "web.h"

/* See comment in date.c.
 */
#ifdef DEBUG
int debug_buf_; /* set at launch time by date:get_sys_time() */

#define DEBUG_BUF_BEFORE if (debug_buf_) {\
	buffer[first] = '\0';\
	(void)fprintf(stderr, "Entering input_ln, First: %d, quoted buf is:\n`%s'\n",\
		first, (char *)buffer+1);\
	}

#define DEBUG_BUF_ADDED if (debug_buf_) (void)fprintf(stderr,\
		"Added in buf after fgets(3) is:\n`%s'\n", (char *)buffer+first)

#define DEBUG_BUF_FINAL if (debug_buf_) {\
	buffer[last] = '\0';\
	(void)fprintf(stderr, "Leaving input_ln, quoted added line is:\n`%s'\n",\
		(char *)buffer+first);\
	}
#else
#define DEBUG_BUF_BEFORE
#define DEBUG_BUF_ADDED
#define DEBUG_BUF_FINAL
#endif /* DEBUG */

/* Read a line of input as efficiently as possible.

   Sets last==first and returns false if eof is hit.

   Otherwise, true is returned and either last==first or buffer[last-1]!=' ';
   that is, last == first + length(line except trailing whitespaces i.e.
   all bytes with isspace(3) true---this includes the eoln whatever
   combination it may actually be).

   The variables declared extern are defined in the WEB sources (and
   declared extern too in the header generated by the translation
   process).
 */
extern unsigned int cbufsize; /* bufsize is a constant; the code calling
	define a variable cbufsize to hold the value of the constant;
	actually, the size of the array is cbufsize+1, the cbufsize index
	is valid but never considered as an input byte */

extern unsigned int maxbufstack, first, last;	/* WEB */
extern unsigned char buffer[];	/* WEB; size cbufsize +1  per pp2rc */
extern unsigned char xord[];	/* WEB */

/* This routine is described in the WEB source in Pascal. Divergence:
   we use a routine to get a whole line, letting the library handling
   whatever means eoln on the system, to have '\n' as prescribed. We
   suppress all the trailing spaces (spaces as by isspace(3)) while our
   xord[] array takes everything as is---spaces "inside" the line will
   not be mapped). [TL]
 */
Boolean
cinputln(FILE *f, Boolean bypass_eoln)
{
    register int i;

	bypass_eoln = false;	/* var ignored: we consume eoln;
	                           keep compiler quiet */

    last = first;

    if (f == stdin) clearerr(stdin);

	DEBUG_BUF_BEFORE;

	if (fgets((char *)buffer+first, cbufsize+1-first, f) != NULL) {
		char *p;

	DEBUG_BUF_ADDED;

		/* C standard says that the end of line for fgets(3) is '\n',
		   whatever the system. MS Windows does the conversion but only
		   in text mode.
		   A line should be ended by a newline. But there are files that
		   end without it. Accept.
		 */
		if ( (p = strchr((char *)buffer+first, '\n')) == NULL
			&& !feof(f) ) {
			/* We didn't get whole line because of lack of buffer space */
			(void)fprintf(stderr,
				"\nUnable to read in the next line---bufsize=%d\n",
				cbufsize);
			(void)fprintf(stderr, "Please ask a wizard to enlarge me.\n");
			exit(EXIT_FAILURE);
		}

		if (p == NULL) { /* point to nul char at end of added line */
			for (p = (char*)buffer+first; *p != '\0'; ++p)
				;
		}	

		/* We ignore every trailing space like character or nul */
		for (last = p - (char *)buffer; last >= first
			 && (isspace(buffer[last]) || buffer[last] == '\0');
			--last)
			;
		++last; /* last is pointing _after_ the valid data; maybe
			last == first now, if the new entered line is empty */

		for (i = first; i < last; ++i) /* last >= first */
				buffer[i] = xord[buffer[i]];

		if (last >= maxbufstack)
			maxbufstack = last;	/* Record keeping */
	
		DEBUG_BUF_FINAL;

		return(true);

	} else {
		if (feof(f))
			return(false);
		else {
			perror("Error when getting next input line");	
			exit(EXIT_FAILURE);
		}
	}

	/* NOT REACHED */
	return(false);
}

